'use strict'

let midiAccess = null
let midiData = {}

const midiLog = []
let midiLogLastTime = 0

const midiCcValues = Array.from({length: 16}, () => Array.from({length: 128}, () => 0))

function midInDeviceNames () {
    return [...midiAccess.inputs.values()].map(input => input.name)
}

function midOutDeviceNames () {
    return [...midiAccess.outputs.values()].map(output => output.name)
}

function midiIsNoteOn (deviceName, channel, note) {
    const volume = midiData[deviceName]?.channels[channel]?.notes[note] || 0
    return volume > 0
}

function midiSendNoteOn (deviceName, channel, note, volume) {
    if (midiAccess?.outputs?.values) {
        const message = [0x90 + channel, note, volume]
        const outputDevice = [...midiAccess.outputs.values()].find(output => output.name === deviceName)
        if (outputDevice) outputDevice.send(message)
    }
}

function midiCcValue (deviceName, channel, cc) {
    if (channel < 0 || channel > 15 || cc < 0 || cc > 127) throw new Error('Out of range')
    const value = midiData[deviceName]?.channels[channel]?.ccs[cc] || 0
    return value > 0
}

function midiGlobalCcValue (channel, cc) {
    if (channel < 0 || channel > 15 || cc < 0 || cc > 127) throw new Error('Out of range')
    return midiCcValues[channel][cc]
}


if (navigator?.requestMIDIAccess) {
    navigator.requestMIDIAccess().then(
        (_midiAccess) => {
            _pushMidiLogMessage('Got MIDI access')
            midiAccess = _midiAccess

            const inputs = [...midiAccess.inputs.values()]
            const outputs = [...midiAccess.outputs.values()]
            inputs.forEach(port => _midiAddPort(port))
            outputs.forEach(port => _midiAddPort(port))

            midiAccess.onstatechange = (event) => {
                const port = event.port

                _pushMidiLogMessage(
                    `[${port.type} ${port.state}]` +
                    ` Name:'${port.name}'` +
                    ` Manufacturer:'${port.manufacturer}'` +
                    ` ID:'${port.id}'` +
                    ` Version:'${port.version}'`);
            
                if (port.state === 'connected') {
                    _midiAddPort(port)
                } else if (port.state === 'disconnected') {
                    _midiRemovePort(port)
                } 
            };
        },
        (msg) => {
            _pushMidiLogMessage(`Failed to get MIDI access: ${msg}`)
        }
    )
}

function _midiAddPort (port) {
    if (port.type === 'input' && port.state === 'connected') {
        const newDevice = { channels: [] }
        for (let channel = 1; channel <= 16; ++channel) {
            const newChannel = {
                notes: {},
                ccs: {}
            }
            newDevice.channels[channel] = newChannel
        }
        midiData[port.name] = newDevice
        port.onmidimessage = _onMidiMessage
    }
}

function _midiRemovePort (port) {
}

function _onMidiMessage (event) {
    const data = event.data
    const deviceName = event.srcElement.name

    let context = '?'
    const dump = event.data.map(x => x.toString()).join(', ')

    const messageType = data[0] & 0xf0
    const channel = (data[0] & 0x0f) + 1
    if (messageType === 0x90) {
        // Note on
        const note = data[1]
        const volume = data[2]
        if (midiData[deviceName]) midiData[deviceName].channels[channel].notes[note] = volume
        context = `NoteOn, ch:${channel}, note:${note}, volume:${volume}`
    } else if (messageType === 0x80) {
        // Note off
        const note = data[1]
        const volume = data[2]
        if (midiData[deviceName]) midiData[deviceName].channels[channel].notes[note] = 0
        context = `NoteOff, ch:${channel}, note:${note}, volume:${volume}`
    } else if (messageType === 0xb0) {
        // Continuous controller
        const cc = data[1]
        const value = data[2]
        midiCcValues[channel-1][cc] = value
        if (midiData[deviceName]) midiData[deviceName].channels[channel].ccs[cc] = value
        context = `CC, ch:${channel}, cc:${cc}, value:${value}`
    }

    _pushMidiLogMessage(`${event.srcElement.name} => ${context}. Raw: ${dump}. Timestamp: ${event.timeStamp}`)
}

function _pushMidiLogMessage (message) {
    midiLog.push(message)
    if (midiLog.length > 20) {
        midiLog.shift()
    }
    midiLogLastTime = window.performance.now()
}
